--[[
  TrialManager.lua
  LabelSync Plugin - Trial Limitation Manager
  
  未認証ユーザーの試用制限を管理
  - 7日間の試用期間
  - 100枚までの写真処理制限
--]]

local LrPrefs = import 'LrPrefs'
local LrDialogs = import 'LrDialogs'

require 'I18n.lua'

local TrialManager = {}

-- 定数
local TRIAL_DAYS = 7
local TRIAL_PHOTO_LIMIT = 100
local PREF_KEY_FIRST_LAUNCH = "labelsync_first_launch"
local PREF_KEY_PHOTOS_PROCESSED = "labelsync_photos_processed"
local PREF_KEY_ACTIVATED = "labelsync_activated"

-- 初回起動日を記録（未設定の場合のみ）
function TrialManager.recordFirstLaunch()
  local prefs = LrPrefs.prefsForPlugin()
  if not prefs[PREF_KEY_FIRST_LAUNCH] then
    prefs[PREF_KEY_FIRST_LAUNCH] = os.time()
  end
end

-- 認証済みかどうかを確認
function TrialManager.isActivated()
  local prefs = LrPrefs.prefsForPlugin()
  return prefs[PREF_KEY_ACTIVATED] == true
end

-- トライアル期間内かどうかを確認
function TrialManager.isWithinTrialPeriod()
  local prefs = LrPrefs.prefsForPlugin()
  local firstLaunch = prefs[PREF_KEY_FIRST_LAUNCH]
  
  if not firstLaunch then
    -- 初回起動、記録して true を返す
    TrialManager.recordFirstLaunch()
    return true
  end
  
  local now = os.time()
  local daysPassed = (now - firstLaunch) / (60 * 60 * 24)
  
  return daysPassed <= TRIAL_DAYS
end

-- トライアル残り日数を取得
function TrialManager.getTrialDaysRemaining()
  local prefs = LrPrefs.prefsForPlugin()
  local firstLaunch = prefs[PREF_KEY_FIRST_LAUNCH]
  
  if not firstLaunch then
    return TRIAL_DAYS
  end
  
  local now = os.time()
  local daysPassed = (now - firstLaunch) / (60 * 60 * 24)
  local remaining = TRIAL_DAYS - daysPassed
  
  return math.max(0, math.floor(remaining))
end

-- 処理済み写真数を取得
function TrialManager.getPhotosProcessed()
  local prefs = LrPrefs.prefsForPlugin()
  return prefs[PREF_KEY_PHOTOS_PROCESSED] or 0
end

-- 処理済み写真数を加算
function TrialManager.addPhotosProcessed(count)
  local prefs = LrPrefs.prefsForPlugin()
  local current = prefs[PREF_KEY_PHOTOS_PROCESSED] or 0
  prefs[PREF_KEY_PHOTOS_PROCESSED] = current + count
end

-- 残り写真数を取得
function TrialManager.getPhotosRemaining()
  return TRIAL_PHOTO_LIMIT - TrialManager.getPhotosProcessed()
end

-- 写真処理が許可されるか確認
function TrialManager.canProcessPhotos(count)
  if TrialManager.isActivated() then
    -- 認証済みでもライセンス期限切れをチェック
    if TrialManager.isLicenseExpired() then
      return false, "license_expired"
    end
    return true, nil
  end
  
  -- トライアル期間チェック
  if not TrialManager.isWithinTrialPeriod() then
    return false, "trial_expired"
  end
  
  -- 写真数チェック
  local remaining = TrialManager.getPhotosRemaining()
  if remaining <= 0 then
    return false, "photo_limit_reached"
  end
  
  if count > remaining then
    return false, "photo_limit_exceeded", remaining
  end
  
  return true, nil
end

-- ライセンス期限切れかどうかを確認（ローカルチェック）
function TrialManager.isLicenseExpired()
  local prefs = LrPrefs.prefsForPlugin()
  local expiresAt = prefs["labelsync_expires_at"]
  
  if not expiresAt or expiresAt == "" then
    -- 有効期限が設定されていない = 永久ライセンス
    return false
  end
  
  -- ISO形式の日時文字列から日付を抽出
  -- 形式: "2026-12-24T12:56:42.554774+00:00"
  local year, month, day = expiresAt:match("^(%d+)-(%d+)-(%d+)")
  if not year then
    return false  -- パースできない場合は期限切れとしない
  end
  
  -- 期限日の終わりまで有効とする（23:59:59）
  local expiresTime = os.time({
    year = tonumber(year),
    month = tonumber(month),
    day = tonumber(day),
    hour = 23,
    min = 59,
    sec = 59
  })
  
  return os.time() > expiresTime
end

-- トライアル制限メッセージを表示
function TrialManager.showTrialLimitMessage(errorType, remaining)
  local title = I18n._("trial_limit_title") or "試用制限"
  local message
  
  if errorType == "trial_expired" then
    message = I18n._("trial_expired_message") or 
      "7日間の試用期間が終了しました。\n引き続きご利用いただくには、ライセンスをご購入ください。"
  elseif errorType == "photo_limit_reached" then
    message = I18n._("photo_limit_reached_message") or 
      "試用版の写真処理上限（100枚）に達しました。\n引き続きご利用いただくには、ライセンスをご購入ください。"
  elseif errorType == "photo_limit_exceeded" then
    message = string.format(
      I18n._("photo_limit_exceeded_message") or 
        "処理できる写真の残り枚数: %d枚\n選択した写真数が上限を超えています。",
      remaining or 0
    )
  elseif errorType == "license_expired" then
    title = I18n._("license_expired_title") or "ライセンス期限切れ"
    message = I18n._("license_expired_message") or 
      "ライセンスの有効期限が切れました。\n引き続きご利用いただくには、ライセンスをご購入ください。"
  else
    message = "試用制限に達しました。"
  end
  
  LrDialogs.message(title, message, "warning")
end

-- トライアル状態の概要を取得
function TrialManager.getTrialStatus()
  if TrialManager.isActivated() then
    return {
      activated = true,
      trialActive = false,
      daysRemaining = nil,
      photosRemaining = nil,
    }
  end
  
  return {
    activated = false,
    trialActive = TrialManager.isWithinTrialPeriod(),
    daysRemaining = TrialManager.getTrialDaysRemaining(),
    photosRemaining = TrialManager.getPhotosRemaining(),
  }
end

return TrialManager
