--[[
  util/Time.lua
  LabelSync Plugin - Time Utility
  
  ISO 日付のパース/フォーマット、タイムスタンプ比較などを提供します。
  Lightroom の dateTimeOriginal (2001/1/1 00:00:00 GMT 起点の秒) と 
  Lua の os.time (1970/1/1 00:00:00 UTC 起点の秒) 間の変換も含みます。
--]]

Time = {}

-- Lightroom epoch (2001-01-01 00:00:00 GMT) と Unix epoch (1970-01-01 00:00:00 UTC) の差（秒）
-- 2001/1/1 - 1970/1/1 = 31年 + 閏年補正
-- 計算: 978307200 秒
local LR_EPOCH_OFFSET = 978307200

-- Lightroom の dateTimeOriginal を Unix timestamp に変換
-- @param lrTime: Lightroom の秒（2001/1/1 起点）
-- @return: Unix timestamp (1970/1/1 起点)
function Time.lrTimeToUnix(lrTime)
  if not lrTime then return nil end
  return lrTime + LR_EPOCH_OFFSET
end

-- Unix timestamp を Lightroom の dateTimeOriginal に変換
-- @param unixTime: Unix timestamp (1970/1/1 起点)
-- @return: Lightroom の秒（2001/1/1 起点）
function Time.unixToLrTime(unixTime)
  if not unixTime then return nil end
  return unixTime - LR_EPOCH_OFFSET
end

-- ISO 8601 形式の日時文字列を Unix timestamp に変換
-- 対応フォーマット例:
--   "2025-12-21 12:05:03"
--   "2025-12-21T12:05:03"
--   "2025-12-21 12:05:03Z"
--   "2025/12/21 12:05:03"
-- @param dateTimeStr: ISO 形式の日時文字列
-- @return: Unix timestamp, または nil
function Time.parseDateTime(dateTimeStr)
  if not dateTimeStr or dateTimeStr == "" then
    return nil
  end
  
  -- スペースや T を正規化
  local normalized = dateTimeStr:gsub("T", " "):gsub("Z", "")
  
  -- スラッシュをハイフンに置換
  normalized = normalized:gsub("/", "-")
  
  -- パターンマッチング: YYYY-MM-DD HH:MM:SS
  local year, month, day, hour, min, sec = normalized:match("(%d%d%d%d)-(%d%d)-(%d%d)%s+(%d%d):(%d%d):(%d%d)")
  
  if not year then
    -- 秒なしパターン: YYYY-MM-DD HH:MM
    year, month, day, hour, min = normalized:match("(%d%d%d%d)-(%d%d)-(%d%d)%s+(%d%d):(%d%d)")
    sec = "00"
  end
  
  if not year then
    return nil
  end
  
  -- os.time で Unix timestamp に変換
  local timeTable = {
    year = tonumber(year),
    month = tonumber(month),
    day = tonumber(day),
    hour = tonumber(hour),
    min = tonumber(min),
    sec = tonumber(sec)
  }
  
  return os.time(timeTable)
end

-- 2つの日付文字列を結合して Unix timestamp に変換
-- @param dateStr: "2025-12-21" or "2025/12/21"
-- @param timeStr: "12:05:03" or "12:05"
-- @return: Unix timestamp, または nil
function Time.parseDateAndTime(dateStr, timeStr)
  if not dateStr or not timeStr then
    return nil
  end
  
  local combined = dateStr .. " " .. timeStr
  return Time.parseDateTime(combined)
end

-- 2つのタイムスタンプの差の絶対値を計算（秒）
-- @param time1: Unix timestamp 1
-- @param time2: Unix timestamp 2
-- @return: 差の絶対値（秒）
function Time.diff(time1, time2)
  if not time1 or not time2 then
    return nil
  end
  return math.abs(time1 - time2)
end

-- Unix timestamp を ISO 8601 形式の文字列に変換
-- @param unixTime: Unix timestamp
-- @return: "YYYY-MM-DD HH:MM:SS" 形式の文字列
function Time.formatDateTime(unixTime)
  if not unixTime then
    return nil
  end
  return os.date("%Y-%m-%d %H:%M:%S", unixTime)
end


return Time
