--[[
  SnitchListParser.lua
  LabelSync Plugin - Selection List Parser
  
  ShutterSnitch Selection List（TAB/COMMA/SEMICOLON 区切り）を解析し、
  Lua table 形式に変換します。
--]]

require 'Log.lua'
require 'Time.lua'

SnitchListParser = {}

-- ヘッダーキーワードのエイリアス
local HEADER_ALIASES = {
  -- 基本フィールド（既存）
  filename = { "filename", "file name", "file", "name" },
  rating = { "rating", "stars", "star" },
  label = { "label", "color label", "colour label", "color", "colour" },
  date = { "date" },
  time = { "time" },
  datetime = { "datetime", "date time", "timestamp", "capture time" },
  
  -- Phase 1: 高優先度フィールド
  title = { "title", "object name", "name", "document title" },
  description = { "description", "caption", "caption/abstract" },
  headline = { "headline" },
  copyright = { "copyright", "copyright notice" },
  keywords = { "keywords", "keyword", "tags" },
  
  -- Phase 2: 位置情報
  city = { "city" },
  state = { "state", "province", "state/province", "province/state", "province / state" },
  country = { "country" },
  location = { "location", "sub location", "sublocation", "sub-location" },
  countryCode = { "country code", "iso country code" },
  gps = { "gps", "geolocation", "geo", "coordinates" },
  gpsAltitude = { "gps altitude", "altitude" },
  
  -- Phase 3: 著作者情報
  creator = { "creator", "author", "image creator name", "byline" },
  credit = { "credit" },
  source = { "source" },
  copyrightUrl = { "copyright url", "copyright info url" },
  jobTitle = { "job title", "creator's job title", "author title" },
  
  -- Phase 4: その他
  instructions = { "instructions", "special instructions" },
  usageTerms = { "usage terms", "rights usage terms" },
  captionWriter = { "caption writer", "writer/editor", "writer", "editor" },
  transmissionRef = { "transmission reference", "original transmission reference" },
}

-- カラーラベルの正規化マップ
local LABEL_NORMALIZATION = {
  -- 英語
  ["red"] = "red",
  ["yellow"] = "yellow",
  ["green"] = "green",
  ["blue"] = "blue",
  ["purple"] = "purple",
  ["none"] = "none",
  
  -- 日本語
  ["赤"] = "red",
  ["黄"] = "yellow",
  ["黄色"] = "yellow",
  ["緑"] = "green",
  ["青"] = "blue",
  ["紫"] = "purple",
  ["なし"] = "none",
  
  -- その他のバリエーション
  ["r"] = "red",
  ["y"] = "yellow",
  ["g"] = "green",
  ["b"] = "blue",
  ["p"] = "purple",
}

-- 文字列分割関数 (空フィールドも保持)
local function split(str, delimiter)
  local result = {}
  local from = 1
  local delim_from, delim_to = string.find(str, delimiter, from, true)
  while delim_from do
    table.insert(result, string.sub(str, from, delim_from - 1))
    from = delim_to + 1
    delim_from, delim_to = string.find(str, delimiter, from, true) 
  end
  table.insert(result, string.sub(str, from))
  return result
end

-- 文字列トリム
local function trim(s)
  if not s then return nil end
  return s:gsub("^%s+", ""):gsub("%s+$", "")
end

-- 区切り文字を検出
-- @param line: ヘッダー行
-- @return: delimiter ("\t" or ";" or ","), または nil
local function detectDelimiter(line)
  if line:match("\t") then
    return "\t"
  elseif line:match(";") then
    return ";"
  elseif line:match(",") then
    return ","
  end
  return nil
end

-- ヘッダー行かどうかを判定
-- @param fields: 分割されたフィールドのテーブル
-- @return: true/false
local function isHeaderRow(fields)
  if #fields < 3 then
    return false
  end
  
  local matchCount = 0
  for _, field in ipairs(fields) do
    local lower = trim(field:lower()):gsub("%s+", " ")
    
    for category, aliases in pairs(HEADER_ALIASES) do
      for _, alias in ipairs(aliases) do
        if lower == alias then
          matchCount = matchCount + 1
          break
        end
      end
    end
  end
  
  -- 2つ以上のヘッダーキーワードが見つかればヘッダー行と判定
  return matchCount >= 2
end

-- ヘッダーフィールド名からカテゴリを特定
-- @param fieldName: フィールド名
-- @return: category, または nil
local function categorizeHeader(fieldName)
  local lower = trim(fieldName:lower()):gsub("%s+", " ")
  
  for category, aliases in pairs(HEADER_ALIASES) do
    for _, alias in ipairs(aliases) do
      if lower == alias then
        return category
      end
    end
  end
  
  return nil
end

-- rating を正規化
local function normalizeRating(ratingStr)
  if not ratingStr or ratingStr == "" then
    return nil
  end
  
  -- 星文字をカウント
  local starCount = 0
  for _ in ratingStr:gmatch("★") do
    starCount = starCount + 1
  end
  
  if starCount > 0 then
    return math.min(starCount, 5)
  end
  
  -- 数字として解釈
  local num = tonumber(ratingStr)
  if num then
    return math.min(math.max(math.floor(num), 0), 5)
  end
  
  return nil
end

-- color label を正規化
local function normalizeLabel(labelStr)
  if not labelStr or labelStr == "" then
    return nil
  end
  
  local lower = labelStr:lower():gsub("%s+", "")
  return LABEL_NORMALIZATION[lower]
end

-- Selection List をパース
-- @param text: Selection List の生テキスト
-- @return: { success = bool, entries = table, error = string }
function SnitchListParser.parse(text)
  if not text or text:gsub("%s+", "") == "" then
    return { success = false, error = "Empty input" }
  end

  -- NULL文字除去（処理の中断防止）
  text = text:gsub("%z", "")

  -- BOM除去 (UTF-8 BOM: EF BB BF)
  if text:byte(1) == 0xEF and text:byte(2) == 0xBB and text:byte(3) == 0xBF then
    text = text:sub(4)
    Log.info("Removed UTF-8 BOM from input.")
  end
  
  -- 行分割（改行コード正規化）
  local lines = {}
  local unified = text:gsub("\r\n", "\n"):gsub("\r", "\n")
  for line in unified:gmatch("[^\n]+") do
    local trimmed = trim(line)
    if trimmed ~= "" then
      table.insert(lines, trimmed)
    end
  end
  
  Log.info("Input lines count: " .. #lines)
  
  if #lines == 0 then
    return { success = false, error = "No lines found" }
  end
  
  -- 区切り文字を検出
  local delimiter = nil
  local headerLine = nil
  local headerIndex = nil
  
  for i, line in ipairs(lines) do
    local testDelimiter = detectDelimiter(line)
    if testDelimiter then
      -- 簡易的なsplitだとquote内区切り文字に対応できないが、
      -- ShutterSnitchの出力は単純なのでまずはsplitで試す
      local fields = split(line, testDelimiter)
      
      if isHeaderRow(fields) then
        delimiter = testDelimiter
        headerLine = line
        headerIndex = i
        break
      end
    end
  end
  
  if not delimiter or not headerLine then
    return { success = false, error = "Header row not found" }
  end
  
  Log.info("Detected delimiter: " .. (delimiter == "\t" and "TAB" or (delimiter == ";" and "SEMICOLON" or "COMMA")))
  Log.info("Header found at line " .. headerIndex)
  
  -- ヘッダーをパース
  local headerFields = split(headerLine, delimiter)
  local headerMap = {}
  for i, field in ipairs(headerFields) do
    local category = categorizeHeader(field)
    if category then
      headerMap[category] = i
    end
  end
  
  -- データ行をパース
  local entries = {}
  local parseErrorCount = 0
  
  for i = headerIndex + 1, #lines do
    local line = lines[i]
    if line and trim(line) ~= "" then
      local fields = split(line, delimiter)
      
      -- フィールド数が極端に少ない場合はスキップせず、可能な限り読み取る
      -- (末尾の空フィールドが省略されるケースがあるため)
      if #fields >= 1 then
        local entry = {}
        
        -- ヘルパー関数: フィールドを取得
        local function getField(category)
          if headerMap[category] and fields[headerMap[category]] then
            local val = trim(fields[headerMap[category]])
            if val and val ~= "" then
              return val
            end
          end
          return nil
        end
        
        -- filename
        entry.filename = getField("filename")
        
        -- rating
        if headerMap.rating then
          entry.rating = normalizeRating(fields[headerMap.rating])
        end
        
        -- label
        if headerMap.label then
          entry.label = normalizeLabel(fields[headerMap.label])
          entry.label_raw = fields[headerMap.label]
        end
        
        -- Phase 1-4 Fields
        entry.title = getField("title")
        entry.description = getField("description")
        entry.headline = getField("headline")
        entry.copyright = getField("copyright")
        entry.keywords = getField("keywords")
        entry.city = getField("city")
        entry.state = getField("state")
        entry.country = getField("country")
        entry.location = getField("location")
        entry.countryCode = getField("countryCode")
        entry.gps = getField("gps")
        entry.gpsAltitude = getField("gpsAltitude")
        entry.creator = getField("creator")
        entry.credit = getField("credit")
        entry.source = getField("source")
        entry.copyrightUrl = getField("copyrightUrl")
        entry.jobTitle = getField("jobTitle")
        entry.instructions = getField("instructions")
        entry.usageTerms = getField("usageTerms")
        entry.captionWriter = getField("captionWriter")
        entry.transmissionRef = getField("transmissionRef")
        
        -- date/time
        local shotAt = nil
        if headerMap.datetime then
          shotAt = Time.parseDateTime(fields[headerMap.datetime])
        elseif headerMap.date and headerMap.time then
          shotAt = Time.parseDateAndTime(fields[headerMap.date], fields[headerMap.time])
        elseif headerMap.date then
          shotAt = Time.parseDateTime(fields[headerMap.date] .. " 00:00:00")
        end
        
        if shotAt then
          entry.shot_at = Time.formatDateTime(shotAt)
          entry.shot_at_unix = shotAt
        end
        
        table.insert(entries, entry)
      else
        parseErrorCount = parseErrorCount + 1
      end
    end
  end
  
  Log.info("Parsed " .. #entries .. " entries successfully. (Skipped/Error: " .. parseErrorCount .. ")")
  
  return {
    success = true,
    entries = entries,
    headerMap = headerMap,
  }
end

return SnitchListParser
