--[[
  PluginInfoProvider.lua
  LabelSync Plugin - Plugin Manager UI Provider
  
  プラグインマネージャーUIを提供し、認証状態の表示と操作を行う
  認証ロジックは Activation.lua に委譲
--]]

local LrView = import 'LrView'
local LrDialogs = import 'LrDialogs'
local LrPrefs = import 'LrPrefs'
local LrColor = import 'LrColor'
local LrHttp = import 'LrHttp'
local LrTasks = import 'LrTasks'
local LrApplication = import 'LrApplication'

-- 共通モジュールのロード
local JSON = require 'JSON.lua'
local Activation = require 'Activation.lua'

-- =============================================================================
-- 定数（Activation.luaから取得）
-- =============================================================================
local API_BASE_URL = Activation.API_BASE_URL
local PLUGIN_VERSION = Activation.PLUGIN_VERSION

-- セクション定義関数
local function sectionsForTopOfDialog(f, propertyTable)
  
  -- 簡易的な翻訳関数
  local function LOC(key, default)
    if not _G.I18n then
      pcall(require, 'I18n.lua')
    end
    
    if _G.I18n and _G.I18n._ then
      return _G.I18n._(key) or default
    end
    return default
  end

  -- 設定の同期 (LrPrefs <-> propertyTable)
  local prefs = LrPrefs.prefsForPlugin()
  local propName = "pluginLanguage"
  
  -- 初期値のロードと検証
  local savedVal = prefs[propName]
  local validLocales = { auto = true, ja = true, en = true, ko = true, fr = true, es = true, de = true, ["zh-Hans"] = true, ["zh-Hant"] = true, da = true }
  if not validLocales[savedVal] then
    savedVal = "auto"
  end
  
  propertyTable[propName] = savedVal
  
  -- アクティベーション状態の初期化
  propertyTable.isActivated = prefs["labelsync_activated"] == true
  propertyTable.activationCode = prefs["labelsync_activation_code"] or ""
  propertyTable.issuedKind = prefs["labelsync_issued_kind"] or ""
  propertyTable.expiresAt = prefs["labelsync_expires_at"] or ""
  propertyTable.statusMessage = ""
  
  -- 言語設定を即時適用
  local function applyLocale(lang)
    local success, Mod = pcall(require, 'I18n.lua')
    if success and Mod then
      if lang == "auto" then
        Mod.detectLocale()
      else
        Mod.setLocale(lang)
      end
    end
  end
  applyLocale(savedVal)
  
  -- 変更の監視と保存
  propertyTable:addObserver(propName, function(prop, key, value)
    if value then
      prefs[propName] = value
      applyLocale(value)
    end
  end)

  return {
    -- 1. Activation Section
    {
      title = LOC("activation_title", "LabelSync – 認証"),
      bind_to_object = propertyTable,  -- セクション全体にバインド
      
      f:row {
        spacing = f:control_spacing(),
        f:static_text {
          title = LOC("activation_label_code", "ライセンスコード:"),
          alignment = 'right',
          width = LrView.share "label_width",
        },
        f:edit_field {
          value = LrView.bind { key = "activationCode", bind_to_object = propertyTable },
          width_in_chars = 25,
          immediate = true,
          enabled = LrView.bind {
            key = "isActivated",
            bind_to_object = propertyTable,
            transform = function(val) return not val end
          },
        },
        f:push_button {
          title = LOC("activation_button_activate", "認証する"),
          enabled = LrView.bind {
            key = "isActivated",
            transform = function(val) return not val end
          },
          action = function()
            LrTasks.startAsyncTask(function()
              local code = propertyTable.activationCode or ""
              local normalized = Activation.normalizeCode(code)
              
              if normalized == "" then
                LrDialogs.message("Activation", LOC("activation_error_invalid_code", "無効なコードです。"), "critical")
                return
              end
              
              -- API リクエスト
              local requestBody = JSON.encode({
                code = normalized,
                device_id = Activation.getDeviceId(),
                plugin_version = PLUGIN_VERSION,
                lr_version = LrApplication.versionString() or "Unknown",
                os = Activation.getOsInfo(),
                locale = "ja"
              })
              
              local responseBody, headers = LrHttp.post(
                API_BASE_URL .. "/api/license/activate",
                requestBody,
                {
                  { field = "Content-Type", value = "application/json" }
                },
                10
              )
              
              if not responseBody then
                LrDialogs.message("Activation", LOC("activation_error_network", "ネットワークエラーが発生しました。"), "critical")
                return
              end
              
              local response = JSON.decode(responseBody)
              
              -- 空レスポンスのチェック
              if response == nil then
                LrDialogs.message("Activation", LOC("activation_error_network", "サーバーからの応答が不正です。"), "critical")
                return
              end
              
              local statusValue = response.status or "nil"
              
              if statusValue == "Activated" then
                -- 成功
                prefs["labelsync_activated"] = true
                prefs["labelsync_activation_code"] = normalized
                
                -- ライセンス情報を保存
                if response.license then
                  prefs["labelsync_issued_kind"] = response.license.issued_kind
                  prefs["labelsync_expires_at"] = response.license.license_expires_at
                  propertyTable.issuedKind = response.license.issued_kind
                  propertyTable.expiresAt = response.license.license_expires_at
                end
                
                propertyTable.isActivated = true
                LrDialogs.message("Activation", LOC("activation_success", "認証に成功しました！"), "info")
              else
                -- エラー
                local errorKey = response.error or "unknown"
                local message = LOC("activation_error_invalid_code", "無効なコードです。")
                
                if errorKey == "limit_exceeded" then
                  message = LOC("activation_error_limit_exceeded", "このライセンスは端末上限に達しています。")
                elseif errorKey == "revoked" then
                  message = LOC("activation_error_revoked", "このライセンスは無効化されています。")
                elseif errorKey == "expired_license" then
                  message = LOC("activation_error_expired", "このライセンスは期限切れです。")
                end
                
                LrDialogs.message("Activation", message, "critical")
              end
            end)
          end,
        },
        -- 認証解除ボタン
        f:push_button {
          title = LOC("activation_button_deactivate", "認証解除"),
          enabled = LrView.bind {
            key = "isActivated",
            bind_to_object = propertyTable,
          },
          action = function()
            LrTasks.startAsyncTask(function()
              -- 確認ダイアログ
              local confirm = LrDialogs.confirm(
                LOC("deactivation_confirm_title", "認証解除の確認"),
                LOC("deactivation_confirm_message", "この端末の認証を解除しますか？\n他の端末で使用できるようになります。"),
                LOC("button_yes", "はい"),
                LOC("button_no", "いいえ")
              )
              
              if confirm ~= "ok" then return end
              
              local code = prefs["labelsync_activation_code"] or ""
              local normalized = Activation.normalizeCode(code)
              
              if normalized == "" then
                LrDialogs.message("Deactivation", "認証されていません。", "critical")
                return
              end
              
              -- API リクエスト
              local requestBody = JSON.encode({
                code = normalized,
                device_id = Activation.getDeviceId(),
                plugin_version = PLUGIN_VERSION,
                lr_version = LrApplication.versionString() or "Unknown",
                os = Activation.getOsInfo(),
                locale = "ja"
              })
              
              local responseBody, headers = LrHttp.post(
                API_BASE_URL .. "/api/license/deactivate",
                requestBody,
                {
                  { field = "Content-Type", value = "application/json" }
                },
                10
              )
              
              if not responseBody then
                LrDialogs.message("Deactivation", LOC("activation_error_network", "ネットワークエラーが発生しました。"), "critical")
                return
              end
              
              local response = JSON.decode(responseBody)
              
              -- 空レスポンスのチェック
              if response == nil then
                LrDialogs.message("Deactivation", LOC("activation_error_network", "サーバーからの応答が不正です。"), "critical")
                return
              end
              
              if response.status == "Deactivated" then
                -- 成功: ローカルの認証状態をクリア
                prefs["labelsync_activated"] = nil
                prefs["labelsync_activation_code"] = nil
                propertyTable.isActivated = false
                propertyTable.activationCode = ""
                LrDialogs.message("Deactivation", LOC("deactivation_success", "認証を解除しました。"), "info")
              else
                LrDialogs.message("Deactivation", response.message or "エラーが発生しました。", "critical")
              end
            end)
          end,
        },
      },
      
      -- ステータス表示
      f:row {
        spacing = f:control_spacing(),
        f:spacer { width = LrView.share "label_width" },
        f:static_text {
          title = LrView.bind {
            keys = { "isActivated", "issuedKind", "expiresAt" },
            bind_to_object = propertyTable,
            operation = function(binder, values, fromTable)
              local isActivated = values.isActivated
              local issuedKind = values.issuedKind or ""
              local expiresAt = values.expiresAt or ""
              
              if not isActivated then
                -- トライアル情報を表示
                local TrialMgr = require 'TrialManager.lua'
                local daysRemaining = TrialMgr.getTrialDaysRemaining()
                local photosProcessed = TrialMgr.getPhotosProcessed()
                local photosRemaining = math.max(0, 100 - photosProcessed)
                
                if daysRemaining > 0 and photosRemaining > 0 then
                  -- トライアル期間内
                  return string.format(
                    LOC("trial_status_format", "未認証（トライアル: 残り%d日 / 残り%d枚）"),
                    daysRemaining, photosRemaining
                  )
                elseif daysRemaining <= 0 then
                  return LOC("trial_expired", "未認証（トライアル期限切れ）")
                else
                  return LOC("trial_limit_reached", "未認証（処理上限に達しました）")
                end
              end
              
              -- 期限切れチェック
              if expiresAt and expiresAt ~= "" then
                local TrialMgr = require 'TrialManager.lua'
                if TrialMgr.isLicenseExpired() then
                  return LOC("license_expired_title", "ライセンス期限切れ")
                end
              end
              
              -- ライセンス種別のラベル
              local kindLabel = ""
              if issuedKind == "paid" then
                kindLabel = LOC("activation_status_activated", "認証済み")
              elseif issuedKind == "free_campaign" then
                kindLabel = LOC("license_kind_campaign", "キャンペーン")
              elseif issuedKind == "trial" then
                kindLabel = LOC("license_kind_trial", "トライアル")
              else
                kindLabel = LOC("activation_status_activated", "認証済み")
              end
              
              -- 有効期限
              if expiresAt and expiresAt ~= "" then
                -- YYYY-MM-DD 形式に変換（ISO形式から）
                local dateStr = expiresAt:match("^(%d+-%d+-%d+)")
                if dateStr then
                  return kindLabel .. " (" .. LOC("license_expires", "有効期限") .. ": " .. dateStr .. ")"
                end
              end
              
              return kindLabel
            end
          },
          text_color = LrView.bind {
            keys = { "isActivated", "expiresAt" },
            bind_to_object = propertyTable,
            operation = function(binder, values, fromTable)
              local isActivated = values.isActivated
              local expiresAt = values.expiresAt or ""
              
              if not isActivated then
                return LrColor('red')
              end
              
              -- 期限切れチェック
              if expiresAt and expiresAt ~= "" then
                local TrialMgr = require 'TrialManager.lua'
                if TrialMgr.isLicenseExpired() then
                  return LrColor(0.9, 0.3, 0.1)  -- オレンジ/赤
                end
              end
              
              return LrColor(0.2, 0.5, 0.9)  -- 青系
            end
          },
          font = "<system/bold>",
          width = 350,  -- 表示幅を確保
        },
      },
      
      -- 公式サイトへのリンク
      f:row {
        spacing = f:control_spacing(),
        f:spacer { width = LrView.share "label_width" },
        f:push_button {
          title = LOC("button_visit_website", "🌐 公式サイトを開く"),
          action = function()
            LrHttp.openUrlInBrowser("https://omnivas.jp/label-sync/")
          end,
        },
      },
    },
    
    -- 2. Settings Section
    {
      title = LOC("settings_section_title", "Settings"),
      
      f:row {
        spacing = f:control_spacing(),
        f:static_text {
          title = LOC("settings_label_language", "Language:"),
          alignment = 'right',
          width = LrView.share "label_width",
        },
        f:popup_menu {
          value = LrView.bind { key = propName, bind_to_object = propertyTable },
          items = {
            { title = LOC("settings_lang_auto", "Auto (System Default)"), value = "auto" },
            { title = "日本語", value = "ja" },
            { title = "English", value = "en" },
            { title = "한국어", value = "ko" },
            { title = "Français", value = "fr" },
            { title = "Español", value = "es" },
            { title = "Deutsch", value = "de" },
            { title = "简体中文", value = "zh-Hans" },
            { title = "繁體中文", value = "zh-Hant" },
            { title = "Dansk", value = "da" },
          },
        },
      },
      
      f:row {
        f:spacer { width = LrView.share "label_width" },
        f:static_text {
          title = LOC("settings_restart_note", "* Reload plugin to apply changes"),
          size = "small",
        },
      },
    },
  }
end

-- テーブルを返す
return {
  sectionsForTopOfDialog = sectionsForTopOfDialog
}
