--[[
  Matcher.lua
  LabelSync Plugin - Photo Matcher (Robust 2-pass approach)
--]]

local LrPathUtils = import 'LrPathUtils'
require 'Log.lua'
require 'Time.lua'

Matcher = {}

local DEFAULT_TOLERANCE_SECONDS = 5

local function getBaseName(filename)
  if not filename then return nil, nil end
  local base = filename:match("(.+)%..+$")
  if not base then base = filename end
  base = base:upper()
  local normalized = base:gsub("^[_%-]?[A-Z]+", "")
  return base, normalized
end

local function getPhotoFilename(photo)
  local path = photo:getRawMetadata("path")
  if not path then return nil end
  return LrPathUtils.leafName(path)
end

local function getPhotoCaptureTime(photo)
  local lrTime = photo:getRawMetadata("dateTimeOriginal")
  if not lrTime then return nil end
  return Time.lrTimeToUnix(lrTime)
end

function Matcher.match(entries, photos, options)
  options = options or {}
  local mode = options.mode or "filename"
  local duplicateHandling = options.duplicateHandling or "first"
  local toleranceSeconds = options.tolerance_seconds or DEFAULT_TOLERANCE_SECONDS
  
  Log.info("Matching started (mode: " .. mode .. ", handling: " .. duplicateHandling .. ")")
  
  -- 1. Index Photos
  local photosByBaseName = {}
  local photosByNormalized = {}
  local photosByTime = {}
  
  for _, photo in ipairs(photos) do
    local filename = getPhotoFilename(photo)
    local captureTime = getPhotoCaptureTime(photo)
    
    -- Filename Indexing
    if filename then
      local baseName, normalized = getBaseName(filename)
      if baseName then
        if not photosByBaseName[baseName] then photosByBaseName[baseName] = {} end
        table.insert(photosByBaseName[baseName], photo)
        if normalized and normalized ~= "" then
          if not photosByNormalized[normalized] then photosByNormalized[normalized] = {} end
          table.insert(photosByNormalized[normalized], photo)
        end
      end
    end
    
    -- Time Indexing
    if captureTime then
      table.insert(photosByTime, { photo = photo, captureTime = captureTime })
    end
  end
  
  -- 2. Preliminary Matching (Pass 1)
  -- 各エントリが「どの写真を欲しがっているか」を収集する
  -- まだ確定はさせない
  
  local entryCandidates = {} -- entry index -> { photo = photo, distance = num, method = str }
  local photoClaimants = {}  -- photoUuid -> list of { entryIndex = i, distance = num }
  local duplicateWarningsSet = {} -- Setで管理して重複を防ぐ
  
  for i, entry in ipairs(entries) do
    local bestMatch = nil
    local matchMethod = nil
    local bestDist = 999999
    
    local entryBaseName, entryNormalized = getBaseName(entry.filename)
    
    if mode == "filename" then
      if entryBaseName and photosByBaseName[entryBaseName] then
        bestMatch = photosByBaseName[entryBaseName][1]
        matchMethod = "filename"
        bestDist = 0
      elseif entryNormalized and entryNormalized ~= "" and photosByNormalized[entryNormalized] then
        bestMatch = photosByNormalized[entryNormalized][1]
        matchMethod = "filename_normalized"
        bestDist = 1
      end
    else
      -- Time Match Logic (Enhanced with Group Sort)
      if entry.shot_at_unix then
        local entryTime = entry.shot_at_unix
        
        -- Tier 1: 厳密一致タイムスタンプでのグループマッチ (ユーザー要望対応)
        -- 同じ時刻のエントリ群と写真群をファイル名ソートして突き合わせる
        
        -- ここで「その時刻のエントリ内での自分の順位」を知る必要がある
        -- 事前に計算していないので、ここで簡易的に計算（少しコストかかるが安全性優先）
        local myRank = 1
        local myGroupSize = 0
        local sameTimeEntries = {}
        
        -- 同じ時刻のエントリを集める
        for _, other in ipairs(entries) do
          if other.shot_at_unix == entryTime then
            table.insert(sameTimeEntries, other)
          end
        end
        
        -- ★警告: 同一時刻に複数のエントリがある場合、そのグループ全員を警告リストに追加
        local skipThisEntry = false  -- ★フラグ追加
        
        if #sameTimeEntries > 1 then
           for _, e in ipairs(sameTimeEntries) do
             local name = e.filename or "unknown"
             duplicateWarningsSet[name] = true
           end
           
           -- ★重要★ skip モードの場合、重複グループは全てスキップ
           if duplicateHandling == "skip" then
             -- bestMatch は nil のまま (マッチさせない)
             skipThisEntry = true  -- ★Tier 2 もスキップするためのフラグ
             Log.info("Skipping duplicate time group (skip mode): " .. (entry.filename or "unknown"))
           else
             -- first モード: ファイル名順でマッチング
             table.sort(sameTimeEntries, function(a, b) 
               return (a.filename or "") < (b.filename or "") 
             end)
             
             for rank, e in ipairs(sameTimeEntries) do
               if e == entry then
                 myRank = rank
                 break
               end
             end
             
             local sameTimePhotos = {}
             for _, pData in ipairs(photosByTime) do
               if pData.captureTime == entryTime then
                 local fname = getPhotoFilename(pData.photo) or ""
                 table.insert(sameTimePhotos, { photo = pData.photo, filename = fname })
               end
             end
             
             if #sameTimePhotos > 0 then
               table.sort(sameTimePhotos, function(a, b)
                 return a.filename < b.filename
               end)
               
               if sameTimePhotos[myRank] then
                 bestMatch = sameTimePhotos[myRank].photo
                 matchMethod = "time_strict_sequence"
                 bestDist = 0
               end
             end
           end
        else
          -- 単独エントリの場合は通常処理
          local sameTimePhotos = {}
          for _, pData in ipairs(photosByTime) do
            if pData.captureTime == entryTime then
              local fname = getPhotoFilename(pData.photo) or ""
              table.insert(sameTimePhotos, { photo = pData.photo, filename = fname })
            end
          end
          
          if #sameTimePhotos > 0 then
            table.sort(sameTimePhotos, function(a, b)
              return a.filename < b.filename
            end)
            
            if sameTimePhotos[1] then
              bestMatch = sameTimePhotos[1].photo
              matchMethod = "time_strict"
              bestDist = 0
            end
          end
        end
        
        -- Tier 2: 厳密一致で見つからなかった場合、従来の Tolerance 探索
        -- ★ただし、重複グループでスキップされた場合は Tier 2 も実行しない
        if not bestMatch and not skipThisEntry then
          local minDiff = toleranceSeconds + 1
          local candidates = {}
          
          for _, pData in ipairs(photosByTime) do
            local diff = math.abs(entryTime - pData.captureTime)
            if diff <= toleranceSeconds then
              if diff < minDiff then
                minDiff = diff
                candidates = { pData.photo }
              elseif diff == minDiff then
                table.insert(candidates, pData.photo)
              end
            end
          end
          
          if #candidates > 0 then
            -- 候補が複数ある場合の分散ロジック (簡易シフト)
            local shiftIndex = ((i - 1) % #candidates) + 1
            bestMatch = candidates[shiftIndex]
            matchMethod = "time"
            bestDist = minDiff
          end
        end
      end
    end
    
    if bestMatch then
      entryCandidates[i] = { photo = bestMatch, distance = bestDist, method = matchMethod }
      local pid = bestMatch:getRawMetadata("uuid")
      if not photoClaimants[pid] then photoClaimants[pid] = {} end
      table.insert(photoClaimants[pid], { entryIndex = i, distance = bestDist })
    end
  end
  
  -- 3. Resolve Conflicts (Pass 2)
  -- ... (以下変更なし)
  -- 競合（1つの写真に複数のエントリ）を解決する
  
  local finalMatches = {}
  local unmatched = {}
  -- local duplicateWarnings = {} -- 前方へ移動済み
  local matchedByFilename = 0
  local matchedByTime = 0
  
  -- どのエントリが勝者かを判定するマップ
  local entryIsWinner = {} -- entryIndex -> bool
  
  -- 写真ごとに競合判定
  for pid, claimantList in pairs(photoClaimants) do
    if #claimantList == 1 then
      -- 競合なし：そのまま採用
      entryIsWinner[claimantList[1].entryIndex] = true
    else
      -- 競合あり！全員を警告リストに追加（重複候補なので）
      for _, c in ipairs(claimantList) do
        local e = entries[c.entryIndex]
        duplicateWarningsSet[e.filename or "unknown"] = true
      end
      
      if duplicateHandling == "first" then
        -- 早い者勝ち（リスト順で最初のエントリを勝者にする）
        -- claimants は entryIndex 順で入っているとは限らないのでソートする
        table.sort(claimantList, function(a, b) return a.entryIndex < b.entryIndex end)
        
        -- 最初の人だけ勝者
        entryIsWinner[claimantList[1].entryIndex] = true
        
        Log.info("Conflict resolved (first): Entry " .. claimantList[1].entryIndex .. " won photo " .. pid)
        
      elseif duplicateHandling == "skip" then
        -- 全員敗者（何もしない）
        Log.info("Conflict resolved (skip): All " .. #claimantList .. " entries skipped for photo " .. pid)
      end
    end
  end
  
  -- 4. Final construction
  for i, entry in ipairs(entries) do
    if entryIsWinner[i] then
      local cand = entryCandidates[i]
      table.insert(finalMatches, {
        entry = entry,
        photo = cand.photo,
        method = cand.method,
        photoId = cand.photo:getRawMetadata("uuid")
      })
      if cand.method and string.find(cand.method, "^time") then
        matchedByTime = matchedByTime + 1 
      else 
        matchedByFilename = matchedByFilename + 1 
      end
    else
      table.insert(unmatched, entry)
    end
  end
  
  -- Setから配列へ変換してソート
  local duplicateWarnings = {}
  for name, _ in pairs(duplicateWarningsSet) do
    table.insert(duplicateWarnings, name)
  end
  table.sort(duplicateWarnings)
  
  -- ★新機能★ 重複グループの詳細情報を構築
  -- 各グループ: { captureTime = "HH:MM:SS", pairs = { {listFile=..., rawFile=...}, ... } }
  local duplicateGroups = {}
  local seenTimeInDup = {}
  local groupIndex = 0
  
  for _, entry in ipairs(entries) do
    if entry.shot_at_unix then
      local t = entry.shot_at_unix
      if not seenTimeInDup[t] then
        -- この時刻のエントリを全て集める
        local sameTimeEntries = {}
        for _, other in ipairs(entries) do
          if other.shot_at_unix == t then
            table.insert(sameTimeEntries, other)
          end
        end
        
        if #sameTimeEntries > 1 then
          -- 重複グループを発見
          groupIndex = groupIndex + 1
          seenTimeInDup[t] = true
          
          -- エントリをファイル名順にソート
          table.sort(sameTimeEntries, function(a, b)
            return (a.filename or "") < (b.filename or "")
          end)
          
          -- 同じ時刻の写真を集めてソート
          local sameTimePhotos = {}
          for _, pData in ipairs(photosByTime) do
            if pData.captureTime == t then
              local fname = getPhotoFilename(pData.photo) or ""
              table.insert(sameTimePhotos, { photo = pData.photo, filename = fname })
            end
          end
          table.sort(sameTimePhotos, function(a, b)
            return a.filename < b.filename
          end)
          
          -- 時刻を人間が読める形式に変換
          local timeStr = os.date("%H:%M:%S", t)
          
          -- ペアリング情報を構築
          local pairs = {}
          for rank, e in ipairs(sameTimeEntries) do
            local rawFile = sameTimePhotos[rank] and sameTimePhotos[rank].filename or "(未マッチ)"
            table.insert(pairs, {
              listFile = e.filename or "unknown",
              rawFile = rawFile,
            })
          end
          
          table.insert(duplicateGroups, {
            groupNum = groupIndex,
            captureTime = timeStr,
            pairs = pairs,
          })
        end
      end
    end
  end
  
  return {
    matched = finalMatches,
    unmatched = unmatched,
    matchedByFilename = matchedByFilename,
    matchedByTime = matchedByTime,
    duplicateWarnings = duplicateWarnings,
    duplicateGroupCount = groupIndex,
    duplicateGroups = duplicateGroups, -- ★新規追加★
  }
end

return Matcher
