--[[
  Main.lua
  LabelSync Plugin - Main Dialog and Control Flow
  
  メインダイアログの表示と、Parser/Matcher/Applier の呼び出しを制御します。
--]]

local LrFunctionContext = import 'LrFunctionContext'
local LrBinding = import 'LrBinding'
local LrDialogs = import 'LrDialogs'
local LrView = import 'LrView'
local LrFileUtils = import 'LrFileUtils'
local LrTasks = import 'LrTasks'
local LrApplication = import 'LrApplication'
local LrProgressScope = import 'LrProgressScope'

-- プラグイン内モジュールの読み込み（SDK ドキュメント準拠）
require 'I18n.lua'
require 'Log.lua'
require 'SnitchListParser.lua'
require 'Matcher.lua'
require 'Applier.lua'
local TrialManager = require 'TrialManager.lua'

-- メインダイアログを表示
local function showMainDialog()
  LrFunctionContext.callWithContext("showMainDialog", function(context)
    local f = LrView.osFactory()
    local props = LrBinding.makePropertyTable(context)
    
    -- プロパティの初期化
    props.selectionListText = ""
    
    -- 統計用プロパティ
    -- 統計用プロパティ (初期値は文字列で "0")
    props.statFilenameMatch = "0"
    props.statFilenameMismatch = "0"
    props.statTimeDuplicateText = string.format(I18n._("stat_duplicate_format") or "%d (%d)", 0, 0)
    props.statTotal = "0"
    
    props.matchMode = "filename" -- "filename" or "time"
    props.duplicateHandling = "first" -- "first" or "skip"
    props.duplicateWarningText = "" -- 重複警告リスト
    props.duplicateCSVData = "" -- CSV形式の重複データ
    
    -- ファイルから読み込むボタンの処理
    local function loadFromFile()
      local filePath = LrDialogs.runOpenPanel({
        title = I18n._("button_load_file"),
        prompt = I18n._("button_load_file"),
        canChooseFiles = true,
        canChooseDirectories = false,
        canCreateDirectories = false,
        allowsMultipleSelection = false,
        fileTypes = { "txt", "csv", "tsv" },
      })
      
      if filePath and #filePath > 0 then
        local path = filePath[1]
        local file = io.open(path, "r")
        if file then
          local content = file:read("*a")
          file:close()
          
          -- BOM (UTF-8) 除去
          if content and string.len(content) >= 3 and 
             content:byte(1) == 0xEF and content:byte(2) == 0xBB and content:byte(3) == 0xBF then
            content = string.sub(content, 4)
            Log.info("Removed BOM from loaded file.")
          end
          
          props.selectionListText = content
          Log.info("Loaded file: " .. path)
        else
          LrDialogs.message(I18n._("error_title"), "Failed to read file: " .. path, "critical")
          Log.error("Failed to read file: " .. path)
        end
      end
    end
    
    -- Preview ボタンの処理
    local function previewMatches()
      -- 入力チェック
      if not props.selectionListText or props.selectionListText:gsub("%s+", "") == "" then
        LrDialogs.message("Label Sync", I18n._("error_empty_input"), "info")
        return
      end
      
      -- パース
      Log.info("Parsing Selection List...")
      local parseResult = SnitchListParser.parse(props.selectionListText)
      
      if not parseResult.success then
        LrDialogs.message("Label Sync", I18n._("error_no_header") .. "\n\n" .. (parseResult.error or ""), "critical")
        return
      end
      
      local entries = parseResult.entries
      local catalog = LrApplication.activeCatalog()
      local targetPhotos = catalog:getMultipleSelectedOrAllPhotos()
      
      if not targetPhotos or #targetPhotos == 0 then
        LrDialogs.message("Label Sync", I18n._("error_no_photos_selected"), "info")
        return
      end
      
      -- マッチング
      local matchResult = Matcher.match(entries, targetPhotos, { 
        mode = props.matchMode,
        duplicateHandling = props.duplicateHandling
      })
      
      -- 統計情報の更新 (UI反映の確実性を高めるため文字列でセット)
      local dupCount = matchResult.duplicateWarnings and #matchResult.duplicateWarnings or 0
      local groupCount = matchResult.duplicateGroupCount or 0
      local filenameMatch = matchResult.matchedByFilename or 0
      local unmatched = #matchResult.unmatched or 0
      
      -- 一度クリアして強制更新を試みる
      props.statFilenameMatch = ""
      props.statFilenameMismatch = ""
      props.statTotal = ""
      LrTasks.yield() -- UIスレッドに処理を返す
      
      props.statFilenameMatch = tostring(filenameMatch)
      if props.matchMode == "filename" then
          props.statFilenameMismatch = tostring(unmatched)
      else
          props.statFilenameMismatch = "0"
      end
      
      -- 撮影時刻重複
      local dupFormat = I18n._("stat_duplicate_format") or "%d (%d)"
      props.statTimeDuplicateText = string.format(dupFormat, dupCount, groupCount)
      
      -- 総リスト数
      props.statTotal = tostring(#entries)
      
      -- 重複マッチリスト表示用テキスト
      if matchResult.duplicateGroups and #matchResult.duplicateGroups > 0 then
        local lines = {}
        local csvLines = { "Group,CaptureTime,ListFile,CatalogRAW" }  -- CSVヘッダー
        
        for _, group in ipairs(matchResult.duplicateGroups) do
          local groupHeader = string.format("[Group %03d] %s", group.groupNum, group.captureTime)
          table.insert(lines, groupHeader)
          
          for _, pair in ipairs(group.pairs) do
            local pairLine = string.format("  %s → %s", pair.listFile, pair.rawFile)
            table.insert(lines, pairLine)
            
            -- CSV用データ
            local csvLine = string.format("%03d,%s,%s,%s", 
              group.groupNum, group.captureTime, pair.listFile, pair.rawFile)
            table.insert(csvLines, csvLine)
          end
          table.insert(lines, "")  -- グループ間の空行
        end
        
        props.duplicateWarningText = table.concat(lines, "\n")
        props.duplicateCSVData = table.concat(csvLines, "\n")  -- CSV形式データを保持
      else
        props.duplicateWarningText = ""
        props.duplicateCSVData = ""
      end
      
      Log.info("Preview stats updated.")
    end
    
    -- Sync 実行処理 (ダイアログ消去後に呼ばれる)
    local function runSyncTask()
      LrFunctionContext.callWithContext("runSyncTask", function(context)
        -- 再度パースとマッチングを行う（安全のため）
        Log.info("Starting Sync task...")
        local parseResult = SnitchListParser.parse(props.selectionListText)
        if not parseResult.success then return end 
        
        local entries = parseResult.entries
        local catalog = LrApplication.activeCatalog()
        local targetPhotos = catalog:getMultipleSelectedOrAllPhotos()
        
        local matchResult = Matcher.match(entries, targetPhotos, { 
          mode = props.matchMode,
          duplicateHandling = props.duplicateHandling
        })
        
        if #matchResult.matched == 0 then
          LrDialogs.message("Label Sync", I18n._("error_no_matches"), "info")
          return
        end
        
        -- トライアル制限チェック
        local photoCount = #matchResult.matched
        local canProcess, errorType, remaining = TrialManager.canProcessPhotos(photoCount)
        
        if not canProcess then
          TrialManager.showTrialLimitMessage(errorType, remaining)
          Log.info("Sync blocked by trial limit: " .. (errorType or "unknown"))
          return
        end
        
        -- Apply
        local progress = LrProgressScope({
          title = "Label Sync - Syncing...",
          functionContext = context,
        })
        
        local success, errorMessage = Applier.apply(catalog, matchResult.matched, {
          overwriteRating = true,
          overwriteLabel = true, 
        }, progress)
        
        progress:setPortionComplete(1, 1) -- 100% にする
        progress:done()
        
        if success then
          -- トライアル制限: 処理した写真数を記録
          if not TrialManager.isActivated() then
            TrialManager.addPhotosProcessed(photoCount)
          end
          
          Log.info("Sync completed: " .. #matchResult.matched)
          LrDialogs.message(I18n._("sync_success_title"), #matchResult.matched .. I18n._("sync_success_message"), "info")
        else
          LrDialogs.message("Label Sync", "Sync failed: " .. (errorMessage or "unknown"), "critical")
        end
      end)
    end
    
    -- ダイアログのコンテンツ構築
    local contents = f:column {
      spacing = f:control_spacing(),
      bind_to_object = props,
      width = 500, 
      
      -- Selection List エリア
      f:row {
        f:static_text { title = I18n._("label_selection_list"), font = "<system/bold>" },
        f:push_button {
          title = I18n._("help_button_title") or "?",
          width = 25,
          action = function()
            local helpContent = f:column {
              spacing = f:control_spacing(),
              f:static_text {
                title = I18n._("help_dialog_content") or "Please enable: Filename, Rating, Label, Capture Time",
                alignment = "left",
                width = 400,
                height_in_lines = 14,
                wrap = true,
              },
            }
            LrDialogs.presentModalDialog {
              title = I18n._("help_dialog_title") or "Setup Guide",
              contents = helpContent,
              actionVerb = "OK",
              cancelVerb = "< exclude >",
            }
          end,
        },
        f:spacer { fill_horizontal = 1 },
        f:push_button { title = I18n._("button_load_file"), action = loadFromFile },
      },
      f:edit_field {
        value = LrView.bind("selectionListText"),
        immediate = true,
        fill_horizontal = 1,
        height_in_lines = 10,
      },
      -- サンプル形式ヒント（placeholderが動作しないため）
      f:static_text {
        title = I18n._("sample_format_hint") or "Sample: Filename | Rating | Label | Capture Time",
        text_color = import 'LrColor'(0.5, 0.5, 0.5),
        font = "<system/small>",
      },
      f:spacer { height = 5 },
      
      -- 設定エリア
      f:column {
        spacing = f:label_spacing(),
        f:row {
          f:static_text { title = I18n._("label_match_mode"), width = LrView.share "config_label", alignment = "right" },
          f:popup_menu {
            value = LrView.bind("matchMode"),
            items = {
              { title = I18n._("match_mode_filename"), value = "filename" },
              { title = I18n._("match_mode_time"), value = "time" },
            },
            width_in_chars = 35,
          },
        },
        f:row {
          f:static_text { title = I18n._("label_duplicate_handling"), width = LrView.share "config_label", alignment = "right" },
          f:popup_menu {
            value = LrView.bind("duplicateHandling"),
            items = {
              { title = I18n._("duplicate_handling_first"), value = "first" },
              { title = I18n._("duplicate_handling_skip"), value = "skip" },
            },
            width_in_chars = 35,
          },
        },
      },
      
      -- ★重要★ 撮影時間マッチモード選択時の警告
      f:static_text {
        visible = LrView.bind {
          keys = { "matchMode" },
          operation = function(binder, values, fromTable)
            return values.matchMode == "time"
          end
        },
        title = I18n._("warning_time_match_mode") or "⚠ Important: Please verify metadata accuracy by checking image numbers with duplicate capture times.",
        text_color = import 'LrColor'(0.8, 0.1, 0.1),
        font = "<system/bold>",
        width = 500,
        height_in_lines = 2,
        wrap = true,
      },
      
      -- プレビューボタン
      f:spacer { height = 8 },
      f:separator { fill_horizontal = 1 },
      f:spacer { height = 8 },
      
      f:row {
        f:spacer { fill_horizontal = 1 },
        f:push_button {
          title = I18n._("button_preview"),
          width = 200, 
          action = function() LrTasks.startAsyncTask(previewMatches) end,
        },
        f:spacer { fill_horizontal = 1 },
      },
      
      f:spacer { height = 15 },
      
      -- 結果表示エリア (2カラム)
      f:row {
        spacing = 20, 
        -- 左: 重複マッチリスト
        f:column {
          width = 300,
          f:group_box {
            title = I18n._("warning_duplicate_list_title"),
            font = "<system/small>",
            fill_horizontal = 1,
            height = 160,
            f:edit_field {
              value = LrView.bind("duplicateWarningText"),
              enabled = true,  -- テキスト選択・コピー可能に
              fill_horizontal = 1,
              fill_vertical = 1,
              height_in_lines = 7, 
            },
          },
          f:push_button {
            title = I18n._("button_save_csv") or "💾 CSV Save",
            fill_horizontal = 1,
            enabled = LrView.bind {
              keys = { "duplicateCSVData" },
              operation = function(binder, values, fromTable)
                return values.duplicateCSVData and values.duplicateCSVData ~= ""
              end
            },
            action = function()
              if props.duplicateCSVData and props.duplicateCSVData ~= "" then
                LrTasks.startAsyncTask(function()
                  local savePath = LrDialogs.runSavePanel({
                    title = I18n._("save_csv_title") or "Save Duplicate Report",
                    requiredFileType = "csv",
                    canCreateDirectories = true,
                  })
                  if savePath then
                    local file = io.open(savePath, "wb")  -- バイナリモードで開く
                    if file then
                      -- BOM (Byte Order Mark) for UTF-8 - Windows Excelで日本語を正しく表示するため
                      file:write("\239\187\191")  -- UTF-8 BOM: EF BB BF
                      file:write(props.duplicateCSVData)
                      file:close()
                      LrDialogs.message("Label Sync", I18n._("msg_saved_csv") or "CSV saved!", "info")
                    end
                  end
                end)
              end
            end,
          },
        },
        
        -- 右: 統計情報
        f:column {
          spacing = 10,
          f:static_text { title = I18n._("stat_section_title") or "Statistics", font = "<system/bold>" },
          f:spacer { height = 8 },
          
          f:row {
            f:static_text { title = I18n._("stat_filename_match"), width = 120, alignment = "right" },
            f:static_text { title = LrView.bind("statFilenameMatch"), font = "<system/bold>", width = 80 },
          },
          f:row {
            f:static_text { title = I18n._("stat_filename_mismatch"), width = 120, alignment = "right" },
            f:static_text { title = LrView.bind("statFilenameMismatch"), width = 80 },
          },
          f:row {
            f:static_text { title = I18n._("stat_time_duplicate"), width = 120, alignment = "right" },
            f:static_text { title = LrView.bind("statTimeDuplicateText"), text_color = import 'LrColor'("orange"), font = "<system/bold>", width = 80 },
          },
          f:separator { fill_horizontal = 1 },
          f:row {
            f:static_text { title = I18n._("stat_total"), width = 120, alignment = "right" },
            f:static_text { title = LrView.bind("statTotal"), font = "<system/bold>", width = 80 },
          },
        },
      },
    }
    
    -- ダイアログ表示
    local result = LrDialogs.presentModalDialog({
      title = I18n._("dialog_title"),
      contents = contents,
      actionVerb = I18n._("button_sync"),
    })
    
    if result == "ok" then
      LrTasks.startAsyncTask(runSyncTask)
    end
  end)
end

-- メインエントリーポイント
showMainDialog()
