--[[
  JSON.lua
  Minimal JSON encoder/decoder for Lightroom SDK
  
  LrHttp で JSON を送受信するためのシンプルな実装
--]]

local JSON = {}

-- JSON エンコード（Lua テーブル → JSON 文字列）
function JSON.encode(value)
  local t = type(value)
  
  if t == "nil" then
    return "null"
  elseif t == "boolean" then
    return value and "true" or "false"
  elseif t == "number" then
    return tostring(value)
  elseif t == "string" then
    -- エスケープ処理
    local escaped = value:gsub('\\', '\\\\')
                         :gsub('"', '\\"')
                         :gsub('\n', '\\n')
                         :gsub('\r', '\\r')
                         :gsub('\t', '\\t')
    return '"' .. escaped .. '"'
  elseif t == "table" then
    -- 配列かオブジェクトかを判定
    local isArray = true
    local n = 0
    for k, _ in pairs(value) do
      n = n + 1
      if type(k) ~= "number" or k ~= n then
        isArray = false
        break
      end
    end
    
    local parts = {}
    if isArray then
      -- 配列
      for _, v in ipairs(value) do
        table.insert(parts, JSON.encode(v))
      end
      return "[" .. table.concat(parts, ",") .. "]"
    else
      -- オブジェクト
      for k, v in pairs(value) do
        if type(k) == "string" then
          table.insert(parts, JSON.encode(k) .. ":" .. JSON.encode(v))
        end
      end
      return "{" .. table.concat(parts, ",") .. "}"
    end
  else
    return "null"
  end
end

-- JSON デコード（JSON 文字列 → Lua テーブル）
-- シンプルな実装：基本的な JSON のみ対応
local decode_scanWhitespace
local decode_scanString
local decode_scanNumber
local decode_scanConstant
local decode_scanArray
local decode_scanObject
local decode_scanValue

local function decode_error(str, idx, msg)
  error("JSON decode error at position " .. idx .. ": " .. msg)
end

decode_scanWhitespace = function(str, idx)
  while idx <= #str do
    local c = str:sub(idx, idx)
    if c == " " or c == "\t" or c == "\n" or c == "\r" then
      idx = idx + 1
    else
      break
    end
  end
  return idx
end

decode_scanString = function(str, idx)
  idx = idx + 1 -- skip opening quote
  local result = ""
  while idx <= #str do
    local c = str:sub(idx, idx)
    if c == '"' then
      return result, idx + 1
    elseif c == '\\' then
      idx = idx + 1
      local escaped = str:sub(idx, idx)
      if escaped == 'n' then result = result .. "\n"
      elseif escaped == 'r' then result = result .. "\r"
      elseif escaped == 't' then result = result .. "\t"
      elseif escaped == '"' then result = result .. '"'
      elseif escaped == '\\' then result = result .. '\\'
      else result = result .. escaped end
      idx = idx + 1
    else
      result = result .. c
      idx = idx + 1
    end
  end
  decode_error(str, idx, "Unterminated string")
end

decode_scanNumber = function(str, idx)
  local startIdx = idx
  local c = str:sub(idx, idx)
  if c == '-' then idx = idx + 1 end
  while idx <= #str do
    c = str:sub(idx, idx)
    if c:match("[0-9%.eE%+%-]") then
      idx = idx + 1
    else
      break
    end
  end
  local numStr = str:sub(startIdx, idx - 1)
  return tonumber(numStr), idx
end

decode_scanConstant = function(str, idx, constant, value)
  if str:sub(idx, idx + #constant - 1) == constant then
    return value, idx + #constant
  end
  decode_error(str, idx, "Expected " .. constant)
end

decode_scanArray = function(str, idx)
  idx = idx + 1 -- skip '['
  local result = {}
  idx = decode_scanWhitespace(str, idx)
  if str:sub(idx, idx) == ']' then
    return result, idx + 1
  end
  while true do
    local value
    value, idx = decode_scanValue(str, idx)
    table.insert(result, value)
    idx = decode_scanWhitespace(str, idx)
    local c = str:sub(idx, idx)
    if c == ']' then
      return result, idx + 1
    elseif c == ',' then
      idx = idx + 1
      idx = decode_scanWhitespace(str, idx)
    else
      decode_error(str, idx, "Expected ',' or ']'")
    end
  end
end

decode_scanObject = function(str, idx)
  idx = idx + 1 -- skip '{'
  local result = {}
  idx = decode_scanWhitespace(str, idx)
  if str:sub(idx, idx) == '}' then
    return result, idx + 1
  end
  while true do
    idx = decode_scanWhitespace(str, idx)
    if str:sub(idx, idx) ~= '"' then
      decode_error(str, idx, "Expected string key")
    end
    local key
    key, idx = decode_scanString(str, idx)
    idx = decode_scanWhitespace(str, idx)
    if str:sub(idx, idx) ~= ':' then
      decode_error(str, idx, "Expected ':'")
    end
    idx = idx + 1
    idx = decode_scanWhitespace(str, idx)
    local value
    value, idx = decode_scanValue(str, idx)
    result[key] = value
    idx = decode_scanWhitespace(str, idx)
    local c = str:sub(idx, idx)
    if c == '}' then
      return result, idx + 1
    elseif c == ',' then
      idx = idx + 1
    else
      decode_error(str, idx, "Expected ',' or '}'")
    end
  end
end

decode_scanValue = function(str, idx)
  idx = decode_scanWhitespace(str, idx)
  local c = str:sub(idx, idx)
  if c == '"' then
    return decode_scanString(str, idx)
  elseif c == '{' then
    return decode_scanObject(str, idx)
  elseif c == '[' then
    return decode_scanArray(str, idx)
  elseif c == 't' then
    return decode_scanConstant(str, idx, "true", true)
  elseif c == 'f' then
    return decode_scanConstant(str, idx, "false", false)
  elseif c == 'n' then
    return decode_scanConstant(str, idx, "null", nil)
  elseif c == '-' or c:match("[0-9]") then
    return decode_scanNumber(str, idx)
  else
    decode_error(str, idx, "Unexpected character: " .. c)
  end
end

function JSON.decode(str)
  if str == nil or str == "" then
    return nil
  end
  local value, _ = decode_scanValue(str, 1)
  return value
end

return JSON
