--[[
  I18n.lua
  LabelSync Plugin - Internationalization
  
  多言語対応のヘルパーモジュール。
  翻訳データは TranslationKeys.lua で一元管理されます。
  
  対応言語:
  - ja: 日本語
  - en: English  
  - ko: 한국어
  - fr: Français
  - es: Español
  - de: Deutsch
  - zh-Hans: 简体中文
  - zh-Hant: 繁體中文
  - da: Dansk
--]]

I18n = {}

-- マスターテーブルを読み込み（Lightroom SDK 互換）
local LrPathUtils = import 'LrPathUtils'
local pluginPath = LrPathUtils.child(_PLUGIN.path, "TranslationKeys.lua")
local TranslationKeys = dofile(pluginPath)

-- サポートされている言語一覧
local SUPPORTED_LOCALES = { "ja", "en", "ko", "fr", "es", "de", "zh-Hans", "zh-Hant", "da" }

-- 現在のロケール（デフォルトは日本語）
local currentLocale = "ja"

-- ロケールを設定する
function I18n.setLocale(locale)
  for _, supported in ipairs(SUPPORTED_LOCALES) do
    if locale == supported then
      currentLocale = locale
      return true
    end
  end
  return false
end

-- 現在のロケールを取得する
function I18n.getCurrentLocale()
  return currentLocale
end

-- サポートされている言語一覧を取得
function I18n.getSupportedLocales()
  return SUPPORTED_LOCALES
end

-- システムロケールを自動検出
function I18n.detectLocale()
  -- 1. Lightroom の Preference (Plugin Manager) から取得を試みる
  local LrPrefs = import 'LrPrefs'
  local prefs = LrPrefs.prefsForPlugin()
  local override = prefs.pluginLanguage
  
  if override and override ~= "auto" then
    if I18n.setLocale(override) then
      return
    end
  end

  -- 2. Lightroomのアプリケーション言語設定を取得 (最も確実)
  local success, LrApplication = pcall(import, 'LrApplication')
  if success and LrApplication and type(LrApplication.language) == "function" then
    local langSuccess, lang = pcall(function() return LrApplication.language() end)
    
    if langSuccess and lang then
      -- サポートしている言語をマッピング
      local langMap = {
        ja = "ja",
        en = "en",
        ko = "ko",
        fr = "fr",
        es = "es",
        de = "de",
        da = "da",
        ["zh-Hans"] = "zh-Hans",
        ["zh-Hant"] = "zh-Hant",
        zh = "zh-Hans",
      }
      
      if langMap[lang] then
        currentLocale = langMap[lang]
        return
      end
      
      -- 部分一致でチェック（例: "en-US" -> "en"）
      for code, locale in pairs(langMap) do
        if lang:match("^" .. code) then
          currentLocale = locale
          return
        end
      end
    end
  end
  
  -- 2.5 LrLocalization でフォールバック (Windows対応)
  local locSuccess, LrLocalization = pcall(import, 'LrLocalization')
  if locSuccess and LrLocalization then
    local currentLang = nil
    
    -- currentLanguage() を試す
    if type(LrLocalization.currentLanguage) == "function" then
      local s, l = pcall(function() return LrLocalization.currentLanguage() end)
      if s and l then currentLang = l end
    end
    
    if currentLang then
      local langMap = {
        ja = "ja", en = "en", ko = "ko", fr = "fr", es = "es", de = "de", da = "da",
        ["zh-Hans"] = "zh-Hans", ["zh-Hant"] = "zh-Hant", zh = "zh-Hans",
      }
      
      if langMap[currentLang] then
        currentLocale = langMap[currentLang]
        return
      end
      
      for code, locale in pairs(langMap) do
        if currentLang:match("^" .. code) then
          currentLocale = locale
          return
        end
      end
    end
  end

  -- 3. それでもだめならOSコマンド（macOSフォールバック）
  local handle = io.popen("defaults read -g AppleLanguages | grep -o '\"[^\"]*\"' | head -1 | tr -d '\"'")
  if handle then
    local lang = handle:read("*a")
    handle:close()
    
    if lang then
      lang = lang:gsub("%s+", "")
      
      if lang:match("^ja") then
        currentLocale = "ja"
      elseif lang:match("^ko") then
        currentLocale = "ko"
      elseif lang:match("^fr") then
        currentLocale = "fr"
      elseif lang:match("^es") then
        currentLocale = "es"
      elseif lang:match("^de") then
        currentLocale = "de"
      elseif lang:match("^da") then
        currentLocale = "da"
      elseif lang:match("^zh%-Hans") or lang:match("^zh%-CN") then
        currentLocale = "zh-Hans"
      elseif lang:match("^zh%-Hant") or lang:match("^zh%-TW") or lang:match("^zh%-HK") then
        currentLocale = "zh-Hant"
      elseif lang:match("^zh") then
        currentLocale = "zh-Hans"
      else
        currentLocale = "en"
      end
    end
  end
end

-- 翻訳文字列を取得する
-- @param key: 翻訳キー
-- @return: 翻訳された文字列（見つからない場合は英語→キー名の順でフォールバック）
function I18n._(key)
  local keyData = TranslationKeys[key]
  
  if keyData then
    -- 現在のロケールで取得
    if keyData[currentLocale] then
      return keyData[currentLocale]
    end
    
    -- フォールバック: 英語を試す
    if keyData["en"] then
      return keyData["en"]
    end
  end
  
  -- 最終フォールバック: nil を返す (呼び出し元でデフォルト値を使うため)
  return nil
end

-- 翻訳キーが存在するか確認
function I18n.hasKey(key)
  return TranslationKeys[key] ~= nil
end

-- 全キーの一覧を取得（デバッグ用）
function I18n.getAllKeys()
  local keys = {}
  for key, _ in pairs(TranslationKeys) do
    table.insert(keys, key)
  end
  table.sort(keys)
  return keys
end

-- 特定のロケールで不足しているキーを検出（デバッグ用）
function I18n.getMissingKeys(locale)
  local missing = {}
  for key, translations in pairs(TranslationKeys) do
    if not translations[locale] then
      table.insert(missing, key)
    end
  end
  table.sort(missing)
  return missing
end

-- 初期化時にロケールを自動検出
I18n.detectLocale()

return I18n
