--[[
  Applier.lua
  LabelSync Plugin - Metadata Applier
  
  マッチした写真にメタデータを適用します。
  catalog:withWriteAccessDo 内で実行されます。
  
  対応フィールド:
  - Rating (★)
  - Label (カラーラベル)
  - Title, Description, Headline, Copyright
  - Keywords
  - 位置情報 (City, State, Country, Location, GPS)
  - 著作者情報 (Creator, Credit, Source)
--]]

local LrApplication = import 'LrApplication'

require 'Log.lua'

Applier = {}

-- GPS文字列をパース
-- @param gpsStr: "35.6895, 139.6917" または "35.6895°N 139.6917°E" など
-- @return: { latitude, longitude } または nil
local function parseGPS(gpsStr)
  if not gpsStr or gpsStr == "" then
    return nil
  end
  
  -- パターン1: "35.6895, 139.6917"
  local lat, lon = gpsStr:match("([%-%.%d]+)%s*[,;]%s*([%-%.%d]+)")
  if lat and lon then
    return { latitude = tonumber(lat), longitude = tonumber(lon) }
  end
  
  -- パターン2: "35.6895°N 139.6917°E"
  lat, lon = gpsStr:match("([%d%.]+)°?%s*[NS]%s*([%d%.]+)°?%s*[EW]")
  if lat and lon then
    local latNum = tonumber(lat)
    local lonNum = tonumber(lon)
    if gpsStr:match("S") then latNum = -latNum end
    if gpsStr:match("W") then lonNum = -lonNum end
    return { latitude = latNum, longitude = lonNum }
  end
  
  return nil
end

-- キーワード文字列をパース
-- @param keywordsStr: "keyword1, keyword2, keyword3"
-- @return: テーブル { "keyword1", "keyword2", "keyword3" }
local function parseKeywords(keywordsStr)
  if not keywordsStr or keywordsStr == "" then
    return {}
  end
  
  local keywords = {}
  -- セミコロン、カンマ、またはパイプで分割
  for kw in keywordsStr:gmatch("[^,;|]+") do
    kw = kw:gsub("^%s+", ""):gsub("%s+$", "") -- trim
    if kw ~= "" then
      table.insert(keywords, kw)
    end
  end
  return keywords
end

-- メタデータを適用するヘルパー関数
-- @param photo: LrPhoto
-- @param fieldName: メタデータフィールド名
-- @param value: 設定する値
-- @param overwrite: 上書きフラグ
-- @param filename: ログ用ファイル名
-- @return: applied (bool)
local function applyField(photo, fieldName, value, overwrite, filename)
  if not value or value == "" then
    return false
  end
  
  local current = photo:getFormattedMetadata(fieldName)
  
  if overwrite or not current or current == "" then
    local success, err = pcall(function()
      photo:setRawMetadata(fieldName, value)
    end)
    
    if success then
      Log.info("Applied " .. fieldName .. " to " .. (filename or "unknown"))
      return true
    else
      Log.warn("Failed to apply " .. fieldName .. ": " .. tostring(err))
      return false
    end
  else
    Log.info("Skipped " .. fieldName .. " for " .. (filename or "unknown") .. " (already has value)")
    return false
  end
end

-- マッチした写真にメタデータを適用
-- @param catalog: LrCatalog オブジェクト
-- @param matchedItems: { entry, photo, method } の配列
-- @param options: { overwriteRating, overwriteLabel, overwriteMetadata }
-- @param progress: LrProgressScope（オプション）
-- @return: success (bool), errorMessage (string)
function Applier.apply(catalog, matchedItems, options, progress)
  options = options or {}
  local overwriteRating = options.overwriteRating ~= false -- デフォルト true
  local overwriteLabel = options.overwriteLabel == true    -- デフォルト false
  local overwriteMetadata = options.overwriteMetadata == true -- デフォルト false
  
  if not matchedItems or #matchedItems == 0 then
    return false, "No matched items to apply"
  end
  
  Log.info("Applying metadata to " .. #matchedItems .. " photos")
  Log.info("Options: rating=" .. tostring(overwriteRating) .. 
           ", label=" .. tostring(overwriteLabel) .. 
           ", metadata=" .. tostring(overwriteMetadata))
  
  local totalItems = #matchedItems
  local appliedCount = 0
  local skippedCount = 0
  local errorCount = 0
  
  -- catalog:withWriteAccessDo 内で実行（timeout パラメータ付き）
  catalog:withWriteAccessDo("LabelSync - Apply Metadata", function(context)
    for i, item in ipairs(matchedItems) do
      local photo = item.photo
      local entry = item.entry
      local filename = entry.filename or "unknown"
      
      if progress then
        progress:setPortionComplete(i, totalItems)
      end
      
      -- ========== 基本フィールド ==========
      
      -- rating を適用
      if entry.rating then
        local currentRating = photo:getRawMetadata("rating")
        
        if overwriteRating or not currentRating or currentRating == 0 then
          photo:setRawMetadata("rating", entry.rating)
          Log.info("Applied rating " .. entry.rating .. " to " .. filename)
          appliedCount = appliedCount + 1
        else
          skippedCount = skippedCount + 1
        end
      end
      
      -- color label を適用
      if entry.label then
        local currentLabel = photo:getRawMetadata("colorNameForLabel")
        
        if overwriteLabel or not currentLabel or currentLabel == "" or currentLabel == "none" then
          photo:setRawMetadata("colorNameForLabel", entry.label)
          Log.info("Applied label " .. entry.label .. " to " .. filename)
          appliedCount = appliedCount + 1
        else
          skippedCount = skippedCount + 1
        end
      end
      
      -- ========== Phase 1: 高優先度フィールド ==========
      
      -- Title
      if entry.title then
        if applyField(photo, "title", entry.title, overwriteMetadata, filename) then
          appliedCount = appliedCount + 1
        end
      end
      
      -- Description (caption)
      if entry.description then
        if applyField(photo, "caption", entry.description, overwriteMetadata, filename) then
          appliedCount = appliedCount + 1
        end
      end
      
      -- Headline
      if entry.headline then
        if applyField(photo, "headline", entry.headline, overwriteMetadata, filename) then
          appliedCount = appliedCount + 1
        end
      end
      
      -- Copyright
      if entry.copyright then
        if applyField(photo, "copyright", entry.copyright, overwriteMetadata, filename) then
          appliedCount = appliedCount + 1
        end
      end
      
      -- Keywords（特殊処理: 追加方式）
      if entry.keywords then
        local keywords = parseKeywords(entry.keywords)
        for _, kw in ipairs(keywords) do
          local success, err = pcall(function()
            local keyword = catalog:createKeyword(kw, {}, true, nil, true)
            if keyword then
              photo:addKeyword(keyword)
            end
          end)
          
          if success then
            Log.info("Added keyword '" .. kw .. "' to " .. filename)
            appliedCount = appliedCount + 1
          else
            Log.warn("Failed to add keyword '" .. kw .. "': " .. tostring(err))
          end
        end
      end
      
      -- ========== Phase 2: 位置情報 ==========
      
      -- City
      if entry.city then
        applyField(photo, "city", entry.city, overwriteMetadata, filename)
      end
      
      -- State/Province
      if entry.state then
        applyField(photo, "stateProvince", entry.state, overwriteMetadata, filename)
      end
      
      -- Country
      if entry.country then
        applyField(photo, "country", entry.country, overwriteMetadata, filename)
      end
      
      -- Location (Sublocation)
      if entry.location then
        applyField(photo, "location", entry.location, overwriteMetadata, filename)
      end
      
      -- ISO Country Code
      if entry.countryCode then
        applyField(photo, "isoCountryCode", entry.countryCode, overwriteMetadata, filename)
      end
      
      -- GPS
      if entry.gps then
        local gpsData = parseGPS(entry.gps)
        if gpsData then
          local success, err = pcall(function()
            photo:setRawMetadata("gps", gpsData)
          end)
          
          if success then
            Log.info("Applied GPS to " .. filename)
            appliedCount = appliedCount + 1
          else
            Log.warn("Failed to apply GPS: " .. tostring(err))
          end
        end
      end
      
      -- GPS Altitude
      if entry.gpsAltitude then
        local altitude = tonumber(entry.gpsAltitude:match("[%-%.%d]+"))
        if altitude then
          applyField(photo, "gpsAltitude", altitude, overwriteMetadata, filename)
        end
      end
      
      -- ========== Phase 3: 著作者情報 ==========
      
      -- Creator
      if entry.creator then
        applyField(photo, "creator", entry.creator, overwriteMetadata, filename)
      end
      
      -- Credit
      if entry.credit then
        applyField(photo, "credit", entry.credit, overwriteMetadata, filename)
      end
      
      -- Source
      if entry.source then
        applyField(photo, "source", entry.source, overwriteMetadata, filename)
      end
      
      -- Copyright URL
      if entry.copyrightUrl then
        applyField(photo, "copyrightInfoUrl", entry.copyrightUrl, overwriteMetadata, filename)
      end
      
      -- Job Title
      if entry.jobTitle then
        applyField(photo, "jobIdentifier", entry.jobTitle, overwriteMetadata, filename)
      end
      
      -- ========== Phase 4: その他 ==========
      
      -- Instructions
      if entry.instructions then
        applyField(photo, "instructions", entry.instructions, overwriteMetadata, filename)
      end
      
      -- Usage Terms
      if entry.usageTerms then
        applyField(photo, "rightsUsageTerms", entry.usageTerms, overwriteMetadata, filename)
      end
      
      -- Caption Writer
      if entry.captionWriter then
        applyField(photo, "descriptionWriter", entry.captionWriter, overwriteMetadata, filename)
      end
      
    end
  end, { timeout = 60 }) -- タイムアウトを延長（フィールド増加のため）
  
  Log.info("Apply complete: " .. appliedCount .. " applied, " .. skippedCount .. " skipped, " .. errorCount .. " errors")
  
  if errorCount > 0 then
    return false, errorCount .. " errors occurred during apply"
  end
  
  return true, nil
end


return Applier
