--[[
  Activation.lua
  LabelSync Plugin - License Activation (v1.3 対応)
  
  バックエンド API に接続してライセンス認証を行います。
  - activate: ライセンスコードをアクティベート
  - validate: ライセンスの有効性をチェック
  
  対応ステータス:
  - Activated: 認証成功
  - invalid_code: コードが無効
  - limit_exceeded: 端末上限超過
  - revoked: ライセンス無効化済み
  - expired_license: 期限切れ（upgrade_url を表示）
--]]

local LrFunctionContext = import 'LrFunctionContext'
local LrBinding = import 'LrBinding'
local LrDialogs = import 'LrDialogs'
local LrView = import 'LrView'
local LrPrefs = import 'LrPrefs'
local LrTasks = import 'LrTasks'
local LrHttp = import 'LrHttp'
local LrSystemInfo = import 'LrSystemInfo'
local LrMD5 = import 'LrMD5'
local LrApplication = import 'LrApplication'

-- 依存モジュール
require 'I18n.lua'
require 'Log.lua'

-- JSONモジュールのロード
local JSON = require 'JSON.lua'

-- =============================================================================
-- 定数・設定
-- =============================================================================

-- バックエンド API URL（本番環境）
local API_BASE_URL = "https://labelsync-backend-389284934090.asia-northeast1.run.app"

-- プリファレンスキー
local PREF_KEY_ACTIVATED = "labelsync_activated"
local PREF_KEY_ACTIVATION_CODE = "labelsync_activation_code"
local PREF_KEY_DEVICE_ID = "labelsync_device_id"
local PREF_KEY_LICENSE_INFO = "labelsync_license_info"  -- SKU, issued_kind など

-- プラグインバージョン
local PLUGIN_VERSION = "0.9.1"

-- =============================================================================
-- ユーティリティ関数
-- =============================================================================

-- デバイス ID を取得（初回生成、以後は保存値を使用）
local function getDeviceId()
  local prefs = LrPrefs.prefsForPlugin()
  if prefs[PREF_KEY_DEVICE_ID] then
    return prefs[PREF_KEY_DEVICE_ID]
  end
  
  -- 新規生成: MAC アドレス + システム情報のハッシュ
  local systemInfo = LrSystemInfo.summaryString() or ""
  local timestamp = tostring(os.time())
  local rawId = systemInfo .. timestamp .. tostring(math.random(1000000))
  local deviceId = LrMD5.digest(rawId)
  
  prefs[PREF_KEY_DEVICE_ID] = deviceId
  return deviceId
end

-- OS 情報を取得
local function getOsInfo()
  local osName = LrSystemInfo.osVersion() or ""
  if osName:find("Mac") or osName:find("mac") then
    return "macOS"
  elseif osName:find("Windows") or osName:find("windows") then
    return "Windows"
  else
    return "Unknown"
  end
end

-- Lightroom バージョンを取得
local function getLrVersion()
  local version = LrApplication.versionString()
  return version or "Unknown"
end

-- アクティベーションコードを正規化（ハイフン削除・大文字化）
local function normalizeCode(code)
  if not code then return "" end
  return code:gsub("[%s%-]", ""):upper()
end

-- フォーマット済みコードを表示用に整形
local function formatCodeForDisplay(code)
  local normalized = normalizeCode(code)
  if #normalized ~= 16 then
    return code
  end
  return normalized:sub(1,4) .. "-" .. normalized:sub(5,8) .. "-" .. normalized:sub(9,12) .. "-" .. normalized:sub(13,16)
end

-- =============================================================================
-- 認証状態の管理
-- =============================================================================

-- 認証状態を確認
local function isActivated()
  local prefs = LrPrefs.prefsForPlugin()
  return prefs[PREF_KEY_ACTIVATED] == true
end

-- 保存されたコードを取得
local function getSavedCode()
  local prefs = LrPrefs.prefsForPlugin()
  return prefs[PREF_KEY_ACTIVATION_CODE] or ""
end

-- ライセンス情報を取得
local function getLicenseInfo()
  local prefs = LrPrefs.prefsForPlugin()
  local infoStr = prefs[PREF_KEY_LICENSE_INFO]
  if infoStr and infoStr ~= "" then
    local success, info = pcall(function() return JSON.decode(infoStr) end)
    if success then return info end
  end
  return nil
end

-- 認証状態を保存
local function saveActivation(code, licenseInfo)
  local prefs = LrPrefs.prefsForPlugin()
  prefs[PREF_KEY_ACTIVATED] = true
  prefs[PREF_KEY_ACTIVATION_CODE] = normalizeCode(code)
  if licenseInfo then
    prefs[PREF_KEY_LICENSE_INFO] = JSON.encode(licenseInfo)
  end
  Log.info("Activation saved")
end

-- 認証をクリア
local function clearActivation()
  local prefs = LrPrefs.prefsForPlugin()
  prefs[PREF_KEY_ACTIVATED] = nil
  prefs[PREF_KEY_ACTIVATION_CODE] = nil
  prefs[PREF_KEY_LICENSE_INFO] = nil
  Log.info("Activation cleared")
end

-- =============================================================================
-- API 通信
-- =============================================================================

-- POST リクエストを送信
local function apiPost(endpoint, body)
  local url = API_BASE_URL .. endpoint
  local jsonBody = JSON.encode(body)
  
  Log.info("API Request: " .. endpoint)
  
  local responseBody, headers = LrHttp.post(
    url,
    jsonBody,
    {
      { field = "Content-Type", value = "application/json" }
    },
    10 -- タイムアウト秒
  )
  
  if not responseBody then
    Log.error("API Error: No response from server")
    return nil, "network_error"
  end
  
  local success, response = pcall(function() return JSON.decode(responseBody) end)
  if not success then
    Log.error("API Error: Invalid JSON response - " .. tostring(response))
    return nil, "parse_error"
  end
  
  -- 空レスポンスのチェック
  if response == nil then
    Log.error("API Error: Empty or invalid response body")
    return nil, "parse_error"
  end
  
  return response, nil
end

-- =============================================================================
-- アクティベーション API
-- =============================================================================

-- ライセンスをアクティベート
local function activateCode(code)
  local normalized = normalizeCode(code)
  
  if #normalized == 0 then
    return {
      success = false,
      status = "invalid_code",
      message = I18n._("activation_error_invalid_code")
    }
  end
  
  local requestBody = {
    code = normalized,
    device_id = getDeviceId(),
    plugin_version = PLUGIN_VERSION,
    lr_version = getLrVersion(),
    os = getOsInfo(),
    locale = I18n.getCurrentLocale()
  }
  
  local response, err = apiPost("/api/license/activate", requestBody)
  
  if err then
    return {
      success = false,
      status = "network_error",
      message = I18n._("activation_error_network")
    }
  end
  
  if response.status == "Activated" then
    return {
      success = true,
      status = "Activated",
      message = I18n._("activation_success"),
      license = response.license,
      activation_limit = response.activation_limit,
      seats_used = response.seats_used
    }
  else
    -- エラーケース
    local errorKey = response.error or "unknown"
    local message = ""
    
    if errorKey == "invalid_code" then
      message = I18n._("activation_error_invalid_code")
    elseif errorKey == "limit_exceeded" then
      message = I18n._("activation_error_limit_exceeded") or 
                "Activation limit exceeded. Please deactivate on another device."
    elseif errorKey == "revoked" then
      message = I18n._("activation_error_revoked") or 
                "This license has been revoked."
    elseif errorKey == "expired_license" then
      message = I18n._("activation_error_expired") or 
                "This license has expired."
    else
      message = response.message or "Unknown error"
    end
    
    return {
      success = false,
      status = errorKey,
      message = message,
      upgrade_url = response.upgrade_url
    }
  end
end

-- ライセンスの有効性を検証（起動時チェック用）
local function validateLicense()
  local savedCode = getSavedCode()
  if savedCode == "" then
    return { valid = false, reason = "no_code" }
  end
  
  local requestBody = {
    code = savedCode,
    device_id = getDeviceId(),
    plugin_version = PLUGIN_VERSION
  }
  
  local response, err = apiPost("/api/license/validate", requestBody)
  
  if err then
    -- ネットワークエラー時はオフライン許容（前回の状態を維持）
    Log.warn("Validation failed due to network error, using cached state")
    return { valid = isActivated(), reason = "offline" }
  end
  
  if response.status == "valid" then
    return { valid = true }
  else
    -- 無効な場合はローカルの認証状態もクリア
    if response.status ~= "offline" then
      clearActivation()
    end
    return {
      valid = false,
      reason = response.status,
      upgrade_url = response.upgrade_url
    }
  end
end

-- =============================================================================
-- UI
-- =============================================================================

-- Activation ダイアログを表示
local function showActivationDialog()
  LrFunctionContext.callWithContext("showActivationDialog", function(context)
    local f = LrView.osFactory()
    local props = LrBinding.makePropertyTable(context)
    
    -- プロパティの初期化
    props.activationCode = formatCodeForDisplay(getSavedCode())
    props.isActivated = isActivated()
    
    if isActivated() then
      local licenseInfo = getLicenseInfo()
      if licenseInfo and licenseInfo.issued_kind == "trial" then
        props.statusMessage = I18n._("activation_status_trial") or "Trial License Active"
      else
        props.statusMessage = I18n._("activation_status_activated")
      end
    else
      props.statusMessage = I18n._("activation_status_not_activated")
    end
    
    props.isProcessing = false
    
    -- Activate ボタンの処理
    local function doActivate()
      if props.isProcessing then return end
      props.isProcessing = true
      
      local code = props.activationCode
      Log.info("Attempting activation...")
      
      local result = activateCode(code)
      
      if result.success then
        saveActivation(code, result.license)
        props.statusMessage = result.message
        props.isActivated = true
        LrDialogs.message(I18n._("activation_title"), result.message, "info")
      else
        props.statusMessage = result.message
        
        -- 期限切れの場合は購入リンクを案内
        if result.status == "expired_license" and result.upgrade_url then
          local action = LrDialogs.confirm(
            I18n._("activation_title"),
            result.message .. "\n\n" .. (I18n._("activation_upgrade_prompt") or "Would you like to purchase a license?"),
            I18n._("button_yes") or "Yes",
            I18n._("button_no") or "No"
          )
          if action == "ok" then
            LrHttp.openUrlInBrowser(result.upgrade_url)
          end
        else
          LrDialogs.message(I18n._("activation_title"), result.message, "warning")
        end
      end
      
      props.isProcessing = false
    end
    
    -- Deactivate ボタンの処理
    local function doDeactivate()
      clearActivation()
      props.statusMessage = I18n._("activation_status_not_activated")
      props.activationCode = ""
      props.isActivated = false
      Log.info("Deactivated")
    end
    
    -- ダイアログのコンテンツ
    local contents = f:column {
      spacing = f:control_spacing(),
      bind_to_object = props,
      
      -- ヘッダー
      f:static_text {
        title = "Label Sync Activation",
        font = "<system/bold>",
      },
      
      f:spacer { height = 10 },
      
      -- 現在のステータス
      f:row {
        f:static_text {
          title = "Status:",
          width = LrView.share("label_width"),
        },
        f:static_text {
          title = LrView.bind("statusMessage"),
          fill_horizontal = 1,
          text_color = LrView.bind {
            key = "isActivated",
            transform = function(value)
              return value and import('LrColor')(0, 0.6, 0) or import('LrColor')(0.6, 0, 0)
            end
          },
        },
      },
      
      f:spacer { height = 10 },
      
      -- Activation Code 入力
      f:row {
        f:static_text {
          title = I18n._("activation_label_code"),
          width = LrView.share("label_width"),
        },
        f:edit_field {
          value = LrView.bind("activationCode"),
          immediate = true,
          width_in_chars = 25,
          enabled = LrView.bind {
            key = "isActivated",
            transform = function(value) return not value end
          },
        },
      },
      
      f:spacer { height = 15 },
      
      -- ボタン
      f:row {
        spacing = f:control_spacing(),
        
        f:push_button {
          title = I18n._("activation_button_activate"),
          enabled = LrView.bind {
            key = "isActivated",
            transform = function(value) return not value end
          },
          action = function()
            LrTasks.startAsyncTask(doActivate)
          end,
        },
        
        f:push_button {
          title = I18n._("activation_button_deactivate") or "Deactivate",
          enabled = LrView.bind("isActivated"),
          action = function()
            doDeactivate()
          end,
        },
      },
      
      f:spacer { height = 10 },
      
      -- デバイス ID 表示（デバッグ用）
      f:static_text {
        title = "Device: " .. getDeviceId():sub(1, 8) .. "...",
        font = "<system/small>",
        text_color = import('LrColor')(0.5, 0.5, 0.5),
      },
    }
    
    -- ダイアログ表示
    LrDialogs.presentModalDialog({
      title = I18n._("activation_title"),
      contents = contents,
    })
  end)
end

-- =============================================================================
-- 公開 API
-- =============================================================================

local Activation = {
  -- 状態確認
  isActivated = isActivated,
  getSavedCode = getSavedCode,
  getLicenseInfo = getLicenseInfo,
  
  -- アクション
  activate = activateCode,
  validate = validateLicense,
  clearActivation = clearActivation,
  showDialog = showActivationDialog,
  
  -- ユーティリティ（PluginInfoProvider等から使用）
  getDeviceId = getDeviceId,
  getOsInfo = getOsInfo,
  getLrVersion = getLrVersion,
  normalizeCode = normalizeCode,
  formatCodeForDisplay = formatCodeForDisplay,
  
  -- 定数
  API_BASE_URL = API_BASE_URL,
  PLUGIN_VERSION = PLUGIN_VERSION,
}

-- スタンドアロン実行用
if _G.arg then
  showActivationDialog()
end

return Activation
